#include "graph.hpp"
#include "cyclic_connectivity.hpp"
#include <iostream>
#include <chrono>
#include <vector>
#include <string>

int main() {
    using namespace std;

    using clock = chrono::high_resolution_clock;

    // Helper to measure and print computation time
    auto measure = [](auto func, const string& name) {
        auto start = clock::now();
        int result = func();
        auto end = clock::now();
        chrono::duration<double> elapsed = end - start;
        cout << name << " computation time: " << elapsed.count() << " s\n";
        return result;
        };


    // Example 1: Complete graph K4 ---
    Graph g1 = Graph::complete(4);
    vector<vector<int>> min_cuts1;
    int ce1 = measure([&]() { return cyclic_edge_connectivity(g1, min_cuts1); }, "Complete K4");
    cout << "Complete K4 cyclic edge connectivity: " << ce1 << "\n\n";



    // Example 2: Petersen graph ---
    Graph g2(10);
    g2.add_edge(0, 1);
    g2.add_edge(0, 5);
    g2.add_edge(1, 2);
    g2.add_edge(1, 6);
    g2.add_edge(2, 3);
    g2.add_edge(2, 7);
    g2.add_edge(3, 4);
    g2.add_edge(3, 8);
    g2.add_edge(4, 1);
    g2.add_edge(4, 9);
    g2.add_edge(5, 7);
    g2.add_edge(6, 8);
    g2.add_edge(7, 9);
    g2.add_edge(8, 5);
    g2.add_edge(9, 6);

    vector<vector<int>> min_cuts2;
    int ce2 = measure([&]() { return cyclic_edge_connectivity(g2, min_cuts2); }, "Petersen graph");
    cout << "Petersen graph cyclic edge connectivity: " << ce2 << "\n\n";



    // Example 3: Large ladder graph 100 vertices ---
    Graph g3(100);
    for (int i = 0; i < 99; ++i) g3.add_edge(i, i + 1);
    for (int i = 0; i < 50; ++i) g3.add_edge(i, i + 50);

    vector<vector<int>> min_cuts3;
    int ce3 = measure([&]() { return cyclic_edge_connectivity(g3, min_cuts3); }, "Large ladder 100");
    cout << "Large ladder 100 cyclic edge connectivity: " << ce3 << "\n\n";



    // Example 4: Grid 10x12 ---
    Graph g4(120);
    int rows = 10, cols = 12;
    for (int r = 0; r < rows; ++r) {
        for (int c = 0; c < cols; ++c) {
            int v = r * cols + c;
            if (c + 1 < cols) g4.add_edge(v, v + 1);
            if (r + 1 < rows) g4.add_edge(v, v + cols);
        }
    }

    vector<vector<int>> min_cuts4;
    int ce4 = measure([&]() { return cyclic_edge_connectivity(g4, min_cuts4); }, "Grid 10x12");
    cout << "Grid 10x12 cyclic edge connectivity: " << ce4 << "\n\n";



    // Example 5: Ladder 6x2 ---
    Graph g5(12);
    for (int i = 0; i < 6; ++i) g5.add_edge(i, i + 6);
    for (int i = 0; i < 5; ++i) {
        g5.add_edge(i, i + 1);
        g5.add_edge(i + 6, i + 7);
    }

    vector<vector<int>> min_cuts5;
    int ce5 = measure([&]() { return cyclic_edge_connectivity(g5, min_cuts5); }, "Ladder 6x2");
    cout << "Ladder 6x2 cyclic edge connectivity: " << ce5 << "\n";

    return 0;
}
